﻿/*  Author:     Ben Hall
 *  File:       AICharacter.cs
 *  Date:       4/28/2011
 *  
 *  Purpose:    AICharacter.cs is an abstract class that extends the GameCharacter class.  It contains
 *              the MakeMove() abstract method that its subclasses must override, and also has various
 *              methods used by its subclasses to decide which skills to use when implementing MakeMove().
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace OnlineGame
{
    public abstract class AICharacter : GameCharacter
    {
        protected double decisionChance;
        protected double rndDecision;
        protected const float lowHealth = 0.25f;
        protected const float halfHealth = 0.50f;
        protected const float highHealth = 0.75f;

        protected int currencyCarried;

        public int GetCurrencyCarried()
        {
            return this.currencyCarried;
        }

        /*  MakeMove() is an abstract method that each subclass should override with a method
         *  containing a decision tree based on the style of AI the subclass represents.
         */
        public abstract String MakeMove();

        /*  DetermineLevel() determines an appropriate level from the AICharacter's stats and
         *  number of skills known.
         *  
         *  Level is determined by adding all stats together and adding points for each skill
         *  known, with offensive skills adding 15 points and defensive skills adding 7 points.
         *  The result is then divided by 30 to get the character's level
         */
        public void DetermineLevel()
        {
            int characterPoints = 0;
            //add up all stat points
            characterPoints += this.strength.stat + this.stamina.stat + this.agility.stat + this.intelligence.stat + this.faith.stat + this.persona.stat;
            
            for (int i = 0; i < this.skillsKnown.Length; i++)
            {
                if (skillsKnown[i] == true)
                {
                    if (MainPage.skillList[i].IsBeneficial() == true)
                    {
                        characterPoints += 7;
                    }
                    else
                    {
                        characterPoints += 15;
                    }
                }
            }
            
            int characterLevel = (int)(characterPoints / 30);
            this.level = characterLevel;
        }

        /*  DetermineDecisionChance() determines the chance that the AICharacter has of picking
         *  its "best" move at each branch of the decision tree.
         *  This chance is based on the AICharacter's intelligence score, with a maximum value of 95%
         */
        protected double DetermineDecisionChance()
        {
            //chance of selecting "best" move is 45 + [floor(INT/10) * 5]
            double decision = 0.45 + ((int)(this.intelligence.stat / 10) * 0.05);

            return decision;
        }

        /*  GetDecisionChance() returns the AICharacter's decisionChance.
         */
        public double GetDecisionChance()
        {
            return this.decisionChance;
        }

        /*  RetermineUseBestMove() adds 0.1 to the chance that an AICharacter
         *  chooses the best choice for itself while following the decision tree for
         *  its AI style.  It also gives rndDecision a new random double value to
         *  compare decisionChance to.
         *  
         *  This method is called in each step in the decision tree after an AI
         *  has skipped the previous step.
         */
        protected Boolean RedetermineUseBestMove()
        {
            //increase chance of selecting next-best move if previous best was ignored
            this.decisionChance += 0.1;
            //create new rndDecision for this step
            this.rndDecision = MainPage.rnd.NextDouble();
            Boolean useBest = (this.rndDecision < this.decisionChance);
            return useBest;
        }

        /*  GetCompHealSkill() returns an integer corresponding to the skill ID of
         *  the best heal skill available to the AICharacter.
         *  
         *  It returns -1 to show that no heal skill is known or available for use.
         */
        protected int GetCompHealSkill()
        {
            int chosenSkill = -1;
            int bestHealValue = 0;
            for (int i = 0; i < MainPage.skillList.Length; i++)
            {
                if (MainPage.skillList[i] != null)
                {
                    if ((MainPage.skillList[i].IsHealing()) && (CanUseSkill(i)))
                    {
                        if (MainPage.skillList[i].GetNumValue() > bestHealValue)
                        {
                            chosenSkill = i;
                            bestHealValue = MainPage.skillList[i].GetNumValue();
                        }
                    }
                }
            }
            return chosenSkill;
        }

        /*  GetCompAttackSkill() returns an integer corresponding to the skill ID of
         *  the best attack skill available to the AICharacter.
         *  
         *  It returns -1 to show that no attack skill is known or available for use.
         */
        protected int GetCompAttackSkill()
        {
            int chosenSkill = -1;
            int bestDamageValue = 0;
            for (int i = 0; i < MainPage.skillList.Length; i++)
            {
                if (MainPage.skillList[i] != null)
                {
                    if ((MainPage.skillList[i].isDamaging()) && (CanUseSkill(i)))
                    {
                        if (MainPage.skillList[i].GetNumValue() > bestDamageValue)
                        {
                            chosenSkill = i;
                            bestDamageValue = MainPage.skillList[i].GetNumValue();
                        }
                    }
                }
            }
            return chosenSkill;
        }
        
        /*  GetCompPDefenseUpSkill() returns an integer corresponding to the skill ID of the
         *  skill available to the AICharacter that increases physical defense the most.
         *  [Causes PDefenseChange with positive value]
         *  
         *  It returns -1 when no physical defense skill is known or available for use.
         */
        protected int GetCompPDefenseUpSkill()
        {
            int chosenSkill = -1;
            float bestPDefenseValue = 0;
            for (int i = 0; i < MainPage.skillList.Length; i++)
            {
                if (MainPage.skillList[i] != null)
                {
                    if ((MainPage.skillList[i].GetPDefenseChange() > bestPDefenseValue) && (CanUseSkill(i)))
                    {
                        chosenSkill = i;
                        bestPDefenseValue = MainPage.skillList[i].GetPDefenseChange();
                    }
                }
            }
            return chosenSkill;
        }

        /*  GetCompPDefenseDownSkill() returns an integer corresponding to the skill ID of the
         *  skill available to the AICharacter that decreases physical defense the most.
         *  [Causes PDefenseChange with negative value]
         *  
         *  It returns -1 when no physical defense reduction skill is known or available for use.
         */
        protected int GetCompPDefenseDownSkill()
        {
            int chosenSkill = -1;
            float bestPDefenseValue = 0;
            for (int i = 0; i < MainPage.skillList.Length; i++)
            {
                if (MainPage.skillList[i] != null)
                {
                    if ((MainPage.skillList[i].GetPDefenseChange() < bestPDefenseValue) && (CanUseSkill(i)))
                    {
                        chosenSkill = i;
                        bestPDefenseValue = MainPage.skillList[i].GetPDefenseChange();
                    }
                }
            }
            return chosenSkill;
        }

        /*  GetCompMDefenseUpSkill() returns an integer corresponding to the skill ID of the
         *  skill available to the AICharacter that increases magical defense the most.
         *  [Causes MDefenseChange with positive value]
         *  
         *  It returns -1 when no magical defense skill is known or available for use.
         */
        protected int GetCompMDefenseUpSkill()
        {
            int chosenSkill = -1;
            float bestMDefenseValue = 0;
            for (int i = 0; i < MainPage.skillList.Length; i++)
            {
                if (MainPage.skillList[i] != null)
                {
                    if ((MainPage.skillList[i].GetMDefenseChange() > bestMDefenseValue) && (CanUseSkill(i)))
                    {
                        chosenSkill = i;
                        bestMDefenseValue = MainPage.skillList[i].GetPDefenseChange();
                    }
                }
            }
            return chosenSkill;
        }

        /*  GetCompMDefenseDownSkill() returns an integer corresponding to the skill ID of the
         *  skill available to the AICharacter that decreases magical defense the most.
         *  [Causes MDefenseChange with negative value]
         *  
         *  It returns -1 when no magical defense reduction skill is known or available for use.
         */
        protected int GetCompMDefenseDownSkill()
        {
            int chosenSkill = -1;
            float bestMDefenseValue = 0;
            for (int i = 0; i < MainPage.skillList.Length; i++)
            {
                if (MainPage.skillList[i] != null)
                {
                    if ((MainPage.skillList[i].GetMDefenseChange() < bestMDefenseValue) && (CanUseSkill(i)))
                    {
                        chosenSkill = i;
                        bestMDefenseValue = MainPage.skillList[i].GetPDefenseChange();
                    }
                }
            }
            return chosenSkill;
        }

        /*  GetCompPoisonSkill() returns an integer corresponding to the skillID of the
         *  cheapest poison inflicting skill available to the AICharacter.
         *  
         *  Returns -1 if the AICharacter does not know or can't use any poisoning skills.
         */
        protected int GetCompPoisonSkill()
        {
            int chosenSkill = -1;
            int lowestCost = 100;
            for (int i = 0; i < MainPage.skillList.Length; i++)
            {
                if (MainPage.skillList[i] != null)
                {
                    if ((MainPage.skillList[i].GetPoisonEffect() > 0) && (CanUseSkill(i)))
                    {
                        if (MainPage.skillList[i].GetSkillCost() < lowestCost)
                        {
                            chosenSkill = i;
                            lowestCost = MainPage.skillList[i].GetSkillCost();
                        }
                    }
                }
            }
            return chosenSkill;
        }

        /*  GetCompAttackSkill() returns an integer corresponding to the skill ID of
         *  the cheapest poison removal skill available to the AICharacter.
         *  
         *  It returns -1 to show that no poison removal skill is known or available for use.
         */
        protected int GetCompRemovePoisonSkill()
        {
            int chosenSkill = -1;
            int lowestCost = 100;
            for (int i = 0; i < MainPage.skillList.Length; i++)
            {
                if (MainPage.skillList[i] != null)
                {
                    if ((MainPage.skillList[i].GetPoisonEffect() < 0) && (CanUseSkill(i)))
                    {
                        if (MainPage.skillList[i].GetSkillCost() < lowestCost)
                        {
                            chosenSkill = i;
                            lowestCost = MainPage.skillList[i].GetSkillCost();
                        }
                    }
                }
            }
            return chosenSkill;
        }
  
        /*  UseRandomDebuff() chooses one skill the AICharacter knows that inflicts some negative status
         *  effect, such as decreasing physical or magical defense, or inflicting poison.
         *  
         *  If no skill exists the default action will be Defend.
         */
        protected String UseRandomDebuff()
        {
            String messageReturned = "";
            int skillChoiceArraySize = 0;
            bool pDefenseDown = false;
            bool mDefenseDown = false;
            bool poison = false;
            if (GetCompPDefenseDownSkill() >= 0)
            {
                skillChoiceArraySize++;
                pDefenseDown = true;
            }
            if (GetCompMDefenseDownSkill() >= 0)
            {
                skillChoiceArraySize++;
                mDefenseDown = true;
            }
            if (GetCompPoisonSkill() >= 0)
            {
                skillChoiceArraySize++;
                poison = true;
            }
            if (skillChoiceArraySize > 0)
            {
                int[] skillChoiceArray = new int[skillChoiceArraySize];
                int arrayPosition = 0;
                if (pDefenseDown == true)
                {
                    skillChoiceArray[arrayPosition] = GetCompPDefenseDownSkill();
                    arrayPosition++;
                }
                if (mDefenseDown == true)
                {
                    skillChoiceArray[arrayPosition] = GetCompMDefenseDownSkill();
                    arrayPosition++;
                }
                if (poison == true)
                {
                    skillChoiceArray[arrayPosition] = GetCompPoisonSkill();
                    arrayPosition++;
                }

                int choice = MainPage.rnd.Next(skillChoiceArraySize);
                //use the chosen skill
                messageReturned = MainPage.UseSkill(MainPage.skillList[skillChoiceArray[choice]], this, MainPage.playerCharacter);
            }
            else
            {
                //default action
                messageReturned = MainPage.DefendCommand(this);
            }
            return messageReturned;
        }

        /*  UseBestAttack() uses the comp's strongest attack move (based on numerical value) available,
         *  defaulting to the Fight command if none is available.
         */
        protected String UseBestAttack()
        {
            String messageReturned = "";
            int chosenSkill = GetCompAttackSkill();
            if (chosenSkill >= 0)
            {
                messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, MainPage.playerCharacter);
            }
            else
            {
                messageReturned = MainPage.FightCommand(this, MainPage.playerCharacter);
            }
            return messageReturned;
        }

        /*  UseBestDefense() uses the comp's best defensive skill available, defaulting to the Defend command
         *  if no skill is available.
         */
        protected String UseBestDefense()
        {
            String messageReturned = "";
            //possible change, take in boolean isDefender.  Only use defensive skills on self unless DefenderAI
            int chosenSkill = -1;
            GameCharacter target = null;
            //50% chance of choosing a physical defense boosting move, 50% chance of magical defense boosting move
            if (MainPage.rnd.NextDouble() < 0.5)
            {
                //Select the best PDefenseUp skill available to the user
                chosenSkill = GetCompPDefenseUpSkill();
                bool targetSelected = false;
                if (chosenSkill >= 0)
                {
                    double randomChance = MainPage.rnd.NextDouble();
                    //AttackerAI's will only use defensive skills on themselves)
                    if (this is AttackerAI)
                    {
                        //determine chance of using a defensive skill, 40% of using skill, 60% of Defend command
                        if ((randomChance < 0.4) && (this.pDefenseChange <= 0))
                        {
                            messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
                        }
                        else
                        {
                            messageReturned = MainPage.DefendCommand(this);
                        }
                    }
                    else
                    {
                        //determine target for chosen skill.  self: 40% chance, non-covered: 25% chance, covered: 15% chance, none(Defend): 10% chance
                        //  If the target already has a PDefenseUp value active, the comp targets the next possible group
                        if ((randomChance < 0.4) && (this.pDefenseChange <= 0))
                        {
                            //use best PDef skill on self
                            target = this;
                            targetSelected = true;
                        }
                        if ((randomChance < 0.65) && (targetSelected == false))
                        {
                            //find ally that is not covered and does not have increased PDef, with lowest health, and use skill
                            for (int i = 0; i < MainPage.compTeam.Length; i++)
                            {
                                //possibly change 0 to the value of the skill's boost
                                if ((MainPage.compTeam[i] != this) && (!MainPage.compTeam[i].IsCovered()) && (MainPage.compTeam[i].GetPDefenseChange() <= 0))
                                {
                                    if (target == null)
                                    {
                                        target = MainPage.compTeam[i];
                                        targetSelected = true;
                                    }
                                    else
                                    {
                                        if (target.GetCurrentHealth() > MainPage.compTeam[i].GetCurrentHealth())
                                        {
                                            target = MainPage.compTeam[i];
                                            targetSelected = true;
                                        }
                                    }
                                }
                            }
                        }
                        if ((randomChance < 0.80) && (targetSelected == false))
                        {
                            //use PDef+ skill on covered ally (if any), if it does not already have increased PDef
                            if (this.IsCoveringCharacter())
                            {
                                //target Covered ally only if it does not have an increased PDef
                                if (this.GetCharacterCovered().GetPDefenseChange() <= 0)
                                {
                                    target = this.GetCharacterCovered();
                                    targetSelected = true;
                                }
                            }
                        }
                        if (targetSelected == false)
                        {
                            //Use Defend command
                            messageReturned = MainPage.DefendCommand(this);
                        }
                        else
                        {
                            //use selected skill on selected target
                            messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, target);
                        }
                    }
                }
                else
                {
                    //use Defend command
                    messageReturned = MainPage.DefendCommand(this);
                }
            }
            else      // use a MDef+ skill, if any.  Default action: Defend
            {
                chosenSkill = GetCompMDefenseUpSkill();
                bool targetSelected = false;
                if (chosenSkill >= 0)
                {
                    double randomChance = MainPage.rnd.NextDouble();
                    if (this is AttackerAI)
                    {
                        //determine chance of using a defensive skill, 50% of using skill, 50% of Defend command
                        if ((randomChance < 0.5) && (this.pDefenseChange <= 0))
                        {
                            messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
                        }
                        else
                        {
                            messageReturned = MainPage.DefendCommand(this);
                        }
                    }
                    else
                    {
                        //determine target for chosen skill.  self: 50% chance, ally: 40% chance, none(Defend): 10% chance
                        //  If the targets already have a MDefenseUp value active, the comp targets the next possible group
                        if ((randomChance < 0.5) && (this.mDefenseChange <= 0))
                        {
                            //use skill on self
                            target = this;
                            targetSelected = true;
                        }
                        if ((randomChance < 0.9) && (targetSelected == false))
                        {
                            //try to find ally without MDef bonus, with preference over ally with lowest health
                            for (int i = 0; i < MainPage.compTeam.Length; i++)
                            {
                                if ((MainPage.compTeam[i] != this) && (MainPage.compTeam[i].GetMDefenseChange() <= 0))
                                {
                                    if (target == null)
                                    {
                                        target = MainPage.compTeam[i];
                                        targetSelected = true;
                                    }
                                    else
                                    {
                                        if (target.GetCurrentHealth() > MainPage.compTeam[i].GetCurrentHealth())
                                        {
                                            target = MainPage.compTeam[i];
                                            targetSelected = true;
                                        }
                                    }
                                }
                            }
                        }
                        if (targetSelected == false)
                        {
                            //use Defend command
                            messageReturned = MainPage.DefendCommand(this);
                        }
                        else
                        {
                            messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, target);
                        }
                    }

                }
                else
                {
                    //use Defend as default
                    messageReturned = MainPage.DefendCommand(this);
                }
            }
            return messageReturned;
        }

    }
}
